/*==============================================================================
Replication package for the BPEA Spring 2024 paper:

"The Emergence of a Uniform Business Cycle in the United States: 
Evidence from New Claims-Based Unemployment Data"

Andrew Fieldhouse, David Munro, Christoffer Koch, and Sean Howard

Stata code for replicating Figures 1-2, Figures 6-8, and Figures 11-12: 
==============================================================================*/

clear all

*Set Your Directory to Replication Package Folder:

cap noi cd "/Users/..../Replication Package_BPEA_CBUR"
u "`c(pwd)'/Rep_Data/Claims_Dataset.dta", clear

bysort Date: egen US_NFE_SA=total(nonfarm_SA)
bysort Date: egen US_IC_SA=total(IC_SA)
bysort Date: egen US_CC_SA=total(CC_SA)

gen IC_CC_Week= (IC_SA+ CC_SA)/MonthtoWeekWeight

gen UR_Claims=(IC_CC_Week)/(IC_CC_Week+ (nonfarm_SA*1000))*100

gen IC_CC_Week_US=(US_IC_SA+ US_CC_SA)/MonthtoWeekWeight

gen UR_Claims_US=(IC_CC_Week_US)/(IC_CC_Week_US+ (US_NFE_SA*1000))*100 if US_NFE_SA>0

sort State Date

gen Date2=mofd(Date) 
format Date2 %tm

xtset fips Date2

gen UR_Claims_US3MA=((F1.IC_CC_Week_US+IC_CC_Week_US+L1.IC_CC_Week_US)/3)/((F1.IC_CC_Week_US+IC_CC_Week_US+L1.IC_CC_Week_US)/3 +(US_NFE_SA*1000))*100 if US_NFE_SA>0

merge m:1 year month using Rep_Data/NBER_Rec.dta, keep(match)
drop _merge
sort State Date
replace USRECM = 15*USRECM

**# Figure 1b: National Comparison URs
twoway area USRECM Date2 if fips==6, color(gs14) blw(none) || line UR_Claims_US3MA USIUR USUR Date2 if fips==6, lc(gs0 gs3 gs8) lp(solid dash dash_dot) bgcolor(white) graphregion(color(white)) xtitle(" ") ytitle("Unemployment Rate") ylabel(0(2)15, nogrid) ymtick(##10) ytitle("") xtitle("") xlab(, nogrid format(%tmCY)) xmtick(##10) legend(order(4 "Official Unemployment Rate" 2 "Claims-Based Unemployment Rate" 3 "Insured Unemployment Rate") pos(11) col(1) ring(0) symxsize(5) size(small)) plotregion(margin(4 4 0 0))

graph export "`c(pwd)'/Rep_Figures/Fig1_URComparison.pdf", replace as(pdf)
export excel Date2 UR_Claims_US3MA USIUR USUR USRECM if fips==6 using "`c(pwd)'/Rep_Figures_Data_Export/Fig1_Export.xlsx", sheet("Fig1b", replace) firstrow(varlabels)

xtset fips Date2
gen UR_Claims_3MA=((F1.IC_CC_Week+IC_CC_Week+L1.IC_CC_Week)/3)/((F1.IC_CC_Week+IC_CC_Week+L1.IC_CC_Week)/3 +(nonfarm_SA*1000))*100 if nonfarm_SA>0
replace USRECM = (17/15)*USRECM
merge 1:1 fips year month using Rep_Data/OH_IUR_NSA_SA.dta
drop if _merge==2

**# Figure 1a: Ohio Comparison URs

twoway area USRECM Date2 if fips==39, color(gs14) || line UR_Claims_3MA OH_IUR UR Date2 if fips==39, lc(gs0 gs3 gs8) lp(solid dash dash_dot) bgcolor(white) graphregion(color(white)) xtitle(" ") ytitle("Unemployment Rate") ylabel(0(2)16, nogrid) ymtick(##10) ytitle("") xtitle("") xlab(, nogrid format(%tmCY)) xmtick(##10) legend(order(4 "Official Unemployment Rate" 2 "Claims-Based Unemployment Rate" 3 "Insured Unemployment Rate") pos(11) col(1) ring(0) symxsize(5) size(small)) plotregion(margin(4 4 0 0))

graph export "`c(pwd)'/Rep_Figures/Fig1_OH_UR.pdf", replace as(pdf)
export excel Date2 UR_Claims_3MA OH_IUR UR USRECM if fips==39 using "`c(pwd)'/Rep_Figures_Data_Export/Fig1_Export.xlsx", sheet("Fig1a", replace) firstrow(varlabels)

drop OH_IUR_NSA OH_IUR_SA_FULL OH_IUR_SA_PRECOVD OH_IUR _merge

**# Generate Fitted CBURs

gen D_ClaimsUR=UR_Claims_3MA-UR_Claims_US3MA

levelsof fips, local(levels)
foreach i of local levels {
di `i'
reg UR USUR D_ClaimsUR if fips==`i' & Date>=date("1/31/1976","MDY")
qui predict UR_Pred_temp_`i' if fips==`i'
corr UR_Pred_temp_`i' UR if fips==`i' & Date>=date("1/31/1976","MDY")
}

egen UR_Pred = rsum(UR_Pred_temp_*)
drop UR_Pred_temp_*
replace UR_Pred=. if UR_Pred==0

**# Figure 2: Ohio fitted plot
twoway area USRECM Date2 if fips==39, color(gs14) || line UR_Claims_3MA UR_Pred UR  Date2 if fips==39, lc(gs8 gs0 gs3) lp(dash solid dash_dot) bgcolor(white) graphregion(color(white)) xtitle(" ") ytitle("Unemployment Rate") ylabel(0(2)16, nogrid) ymtick(##10) ytitle("") xtitle("") xlab(, nogrid format(%tmCY)) xmtick(##10) legend(order(4 "Official UR" 2 "Claims-Based UR" 3 "Claims-Based UR-Fitted") pos(11) col(1) ring(0) symxsize(5) size(small)) plotregion(margin(4 4 0 0))

graph export "`c(pwd)'/Rep_Figures/Fig2_OH_UR_Fit.pdf", replace as(pdf)
export excel Date2 UR_Claims_3MA UR_Pred UR USRECM if fips==39 using "`c(pwd)'/Rep_Figures_Data_Export/Fig2_Export.xlsx", sheet("Fig2", replace) firstrow(varlabels)

**# Figure 6: National Recovery Rates
clear
input int id double us_recov_ur double us_recov_cbur double hk_2022
1	0.3210899	0.3077883	0.31
2	0.1999824	0.1887186	0.20
3	0.2818656	0.3410543	0.28
4	0.1004072	0.1439492	0.09
5	0.0996115	0.1226862	0.13
6	0.1186145	0.188721	0.12
7	0.121596	0.1374619	0.12
8	0.0918029	0.0837577	0.09
9	0.1076835	0.1196177	0.09
10	0.1058644	0.1339282	0.10
11	0.3780803	0.564515	.
end 

twoway scatter us_recov_ur id, msize(vlarge) msymbol(T) mc(gs3%50) mlc(none) || scatter us_recov_cbur id, msize(vlarge) msymbol(+) mc(gs0) || scatter hk_2022 id, msize(vlarge) msymbol(Oh) mc(gs7%50) xlabel(1 "1949" 2 "1954" 3 "1958" 4 "1961" 5 "1970" 6 "1975" 7 "1982" 8 "1992" 9 "2003" 10 "2009" 11 "2020", nogrid) legend(order(2 "Claims-Based Unemployment Rate" 1 "Official Unemployment Rate" 3 "Hall and Kudlyak (2020) Recovery Rates") pos(12) margin(0 0 0 0) ring(0) col(1)) xtitle(" " "Recovery Cycle") subtitle("Recovery Pace", span ring(1) pos(11) size(medsmall) margin(0 0 3 0)) ylab(0(0.1)0.6, format(%4.1fc) nogrid) ymtick(##10) plotregion(m(6 6 0 0)) graphregion(color(white)) bgcolor(white)

graph export "`c(pwd)'/Rep_Figures/Fig6_RecovAgg.pdf", replace as(pdf)
export excel id us_recov_ur us_recov_cbur hk_2022 using "`c(pwd)'/Rep_Figures_Data_Export/Fig6_Export.xlsx", sheet("Fig6", replace) firstrow(varlabels)

**# Figure 7: State-level recovery rates
import delimited "`c(pwd)'/Rep_Data/State_Level_Recoveries.csv", varnames(1) clear

twoway scatter ak_recov id if inrange(ak_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter al_recov id if inrange(al_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter ar_recov id if inrange(ar_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter az_recov id if inrange(az_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter ca_recov id if inrange(ca_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter co_recov id if inrange(co_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter ct_recov id if inrange(ct_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter dc_recov id if inrange(dc_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter de_recov id if inrange(de_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter fl_recov id if inrange(fl_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter ga_recov id if inrange(ga_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter hi_recov id if inrange(hi_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter ia_recov id if inrange(ia_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter id_recov id if inrange(id_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter il_recov id if inrange(il_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter in_recov id if inrange(in_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter ks_recov id if inrange(ks_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter ky_recov id if inrange(ky_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter la_recov id if inrange(la_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter ma_recov id if inrange(ma_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter md_recov id if inrange(md_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter me_recov id if inrange(me_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter mi_recov id if inrange(mi_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter mn_recov id if inrange(mn_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter mo_recov id if inrange(mo_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter ms_recov id if inrange(ms_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter mt_recov id if inrange(mt_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter nc_recov id if inrange(nc_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter nd_recov id if inrange(nd_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter ne_recov id if inrange(ne_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter nh_recov id if inrange(nh_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter nj_recov id if inrange(nj_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter nm_recov id if inrange(nm_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter nv_recov id if inrange(nv_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter ny_recov id if inrange(ny_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter oh_recov id if inrange(oh_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter ok_recov id if inrange(ok_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter or_recov id if inrange(or_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter pa_recov id if inrange(pa_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter ri_recov id if inrange(ri_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter sc_recov id if inrange(sc_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter sd_recov id if inrange(sd_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter tn_recov id if inrange(tn_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter tx_recov id if inrange(tx_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter ut_recov id if inrange(ut_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter va_recov id if inrange(va_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter vt_recov id if inrange(vt_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter wa_recov id if inrange(wa_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter wi_recov id if inrange(wi_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter wv_recov id if inrange(wv_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter wy_recov id if inrange(wy_recov,0,1), msize(medium) mc(gs0%15) mlw(none) || scatter us_recov_ur id, msize(vlarge) msymbol(+) mc(gs0) xlabel(1 "1949" 2 "1954" 3 "1958" 4 "1961" 5 "1970" 6 "1975" 7 "1982" 8 "1992" 9 "2003" 10 "2009" 11 "2020", nogrid) legend(order(50 "State Level Heterogeneity" 52 "U.S. National Level") pos(12) ring(0) col(1)) xtitle(" " "Recovery Cycle") subtitle("Recovery Pace", span ring(1) pos(11) size(medsmall) margin(0 0 3 0)) ylab(0(0.1)0.8, nogrid format(%4.1fc)) ymtick(##10) plotregion(m(6 6 0 0)) graphregion(color(white)) bgcolor(white)

graph export "`c(pwd)'/Rep_Figures/Fig7_RecovStates.pdf", replace as(pdf)

foreach usState in ak al ar az ca co ct dc de fl ga hi ia id il in ks ky la ma md me mi mn mo ms mt nc nd ne nh nj nm nv ny oh ok or pa ri sc sd tn tx ut va vt wa wi wv wy {
	di "`state'"
	replace `usState'_recov = . if !inrange(`usState'_recov,0,1)
}
export excel recoveryyearlabel id *reco* using "`c(pwd)'/Rep_Figures_Data_Export/Fig7_Export.xlsx", sheet("Fig7", replace) firstrow(varlabels)


**# Figure 8: Share of states in recession
*Need Fred API for this
clear frames
frame create nber_rec
frame change nber_rec
import fred USRECM, clear
gen m_date = mofd(daten)

frame change default

import delimited "`c(pwd)'/Rep_Data/Share_Recessions.csv", clear 

gen d_date = date(dater,"MDY",2022), before(dater)
format d_date %td
gen m_date = mofd(d_date), before(dater)

format m_date %tm
frlink 1:1 m_date, frame(nber_rec m_date) generate(fr_link)
frget USRECM, from(fr_link)

twoway bar USRECM m_date, fc(gs5%60) blw(none) || bar sharerecession m_date, fc(gs0%60) blw(none) xtitle(" ") ytitle("") subtitle("Share of States in Recession", span ring(1) pos(11) size(small) margin(0 0 3 0)) ylab(0(0.2)1.0, nogrid format(%4.1fc)) ymtick(##10) xlab(, nogrid format(%tmCY)) xmtick(##10) yline(0, lc(gs12) lw(thin)) legend(order(1 "NBER Recessions" 2 "Share of Recession States") pos(1) ring(1) size(*0.7) region(lcolor(white)) nobox) graphregion(color(white)) bgcolor(white)

graph export "`c(pwd)'/Rep_Figures/Fig8_ShareRec.pdf", replace as(pdf)
export excel m_date sharerecession USRECM using "`c(pwd)'/Rep_Figures_Data_Export/Fig8_Export.xlsx", sheet("Fig8", replace) firstrow(varlabels)

**# Figure 12: Manufacturing shares and recovery pace
import delimited "`c(pwd)'/Rep_Data/ManuShares_RecovPace.csv", clear 

reg recov_pace manufacturingshares if timeperiod == "1948-1957" & inrange(recov_pace,-0.3,1), cformat(%6.4fc)

twoway scatter recov_pace manufacturingshares if timeperiod == "1948-1957" & inrange(recov_pace,-0.3,1), msymbol(o) mc(gs0%75) mlc(none) ylab(-0.3(0.1)0.7, format(%4.1fc) nogrid) ymtick(##10) xlab(0(0.1)0.5, nogrid format(%4.1fc)) xmtick(##10) plotregion(m(0 0 0 0)) yline(0, lc(gs12) lw(thin)) || lfit recov_pace manufacturingshares if timeperiod == "1948-1957", lc(gs0%50) lw(medthick) lp(solid) legend(order(2 "Linear Fit") pos(11) region(style(none)) ring(0) col(1) symxsize(5)) text(0.5 0.1 "y = 0.0331 + 0.7465x", placement(3) c(gs0%75)) xtitle(" " "Manufacturing Shares") ytitle("") subtitle("Mean Pace of Recession Recovery", span ring(1) pos(11) size(medsmall) margin(0 0 3 0)) xsize(3) ysize(3) bgcolor(white) graphregion(color(white))

graph export "`c(pwd)'/Rep_Figures/Fig12_ManuShare4957.pdf", replace as(pdf)
export excel recov_pace manufacturingshares if timeperiod == "1948-1957" & inrange(recov_pace,-0.3,1) using "`c(pwd)'/Rep_Figures_Data_Export/Fig12_Export.xlsx", sheet("Fig12a", replace) firstrow(varlabels)

reg recov_pace manufacturingshares if timeperiod == "1959-2009" & inrange(recov_pace,-0.3,1), cformat(%6.4fc)

scatter recov_pace manufacturingshares if timeperiod == "1959-2009" & inrange(recov_pace,-0.3,1), msymbol(o) mc(gs0%75) mlc(none) ylab(-0.3(0.1)0.7, format(%4.1fc) nogrid) ymtick(##10) xlab(0(0.1)0.5, nogrid format(%4.1fc)) xmtick(##10) plotregion(m(0 0 0 0)) yline(0, lc(gs12) lw(thin)) || lfit recov_pace manufacturingshares if timeperiod == "1959-2009", lc(gs0%50) lw(medthick) lp(solid) legend(order(2 "Linear Fit") pos(11) region(style(none)) ring(0) col(1) symxsize(5)) text(0.3 0.1 "y = 0.0901 + 0.3873x", placement(3) c(gs0%75)) xtitle(" " "Manufacturing Shares") ytitle("") subtitle("Mean Pace of Recession Recovery", span ring(1) pos(11) size(medsmall) margin(0 0 3 0)) xsize(3) ysize(3) bgcolor(white) graphregion(color(white))

graph export "`c(pwd)'/Rep_Figures/Fig12_ManuShare5809.pdf", replace as(pdf)
export excel recov_pace manufacturingshares if timeperiod == "1959-2009" & inrange(recov_pace,-0.3,1) using "`c(pwd)'/Rep_Figures_Data_Export/Fig12_Export.xlsx", sheet("Fig12b", replace) firstrow(varlabels)

**# Figure 11: Differences in state-level industry composition
use "`c(pwd)'/Rep_Data/SAINC5_1929_2022.dta", clear
bysort year: egen SumEarnUSA=sum(gdp_ind_group_usd)
bysort match_id year: egen SumIndEarnUSA=sum(gdp_ind_group_usd)
bysort fips year: egen SumEarn=sum(gdp_ind_group_usd)
gen Ind_Share=gdp_ind_group_usd/SumEarn
gen Ind_ShareUSA=SumIndEarnUSA/SumEarnUSA
gen Ind_Share_Diff2=(Ind_Share-Ind_ShareUSA)^2
bysort fips year: egen SumSquaredDiff=sum(Ind_Share_Diff2)
bysort year: egen Avg_SumSquaredDiff=mean(SumSquaredDiff)
by year: egen WTAvg_SumSquaredDiff = wtmean(SumSquaredDiff), weight(SumEarn)

twoway (line Avg_SumSquaredDiff WTAvg_SumSquaredDiff year if fips==1 & year>=1948 & year<=2001, lc(gs0 gs3) lp(solid shortdash)) (line Avg_SumSquaredDiff WTAvg_SumSquaredDiff year if fips==1 & year>=2002, msymbol(T) lc(gs0 gs3) lp(longdash_dot shortdash_dot)), ///
bgcolor(white) xtitle(" ") ytitle("Avg. Sum of Squared Differences in State Industry Shares") ///
graphregion(color(white)) ymtick(##10) xmtick(##10) ///
ytitle("") xtitle("") yscale(range(0 0.05)) ///
legend(order(2 "Weighted Average - SIC" 1 "Raw Average - SIC" 4 "Weighted Average - NAICS" 3 "Raw Average - NAICS") pos(1) col(1) ring(0) symxsize(10) size(small)) plotregion(margin(4 4 0 0)) ///
subtitle("Avg. Sum of Squared Differences in State Industry Shares", span ring(1) pos(11) size(medsmall) margin(0 0 3 0))

graph export "`c(pwd)'/Rep_Figures/Fig11_IndustryDiff.pdf", replace as(pdf)
export excel year Avg_SumSquaredDiff WTAvg_SumSquaredDiff if fips==1 & year>=1948 using "`c(pwd)'/Rep_Figures_Data_Export/Fig11_Export.xlsx", sheet("Fig11", replace) firstrow(varlabels)

// END OF FILE //
